defmodule Console.Interface do
  @moduledoc """
  Un serveur générique créé de façon unique par application.
  Ce serveur est en charge des écrans, c'est-à-dire des différentes opérations possibles par le joueur en fonction de son expérience de jeu actuelle.
  """

  use GenServer

  @doc """
  Lance l'interface.
  """
  def start_link(options \\ []) do
    options = Keyword.put(options, :name, :interface)
    GenServer.start_link(__MODULE__, nil, options)
  end

  @doc """
  Gère l'entrée clavier de l'utilisateur.
  L'entrée clavier est à préciser sous la forme d'une chaîne.
  """
  @spec gérer_entrée(String.t()) :: :ok
  def gérer_entrée(entrée) do
    GenServer.call(:interface, {:entrée, entrée})
  end

  @impl true
  def init(_) do
    écran = Console.Ecran.NomJoueur.new()

    if Jeu.peut_communiquer?() do
      afficher_retour(écran, :rafraîchir)
      {:ok, écran}
    else
      IO.puts("Impossible de contacter une application 'carte'.")
      {:stop, écran}
    end
  end

  @impl true
  def handle_call({:entrée, entrée}, _from, écran) do
    module = écran.__struct__
    retour = module.gérer_entrée(écran, entrée)
    écran = afficher_retour(écran, retour)
    {:reply, :ok, écran}
  end

  @impl true
  def handle_info(:actualiser, écran) do
    écran = afficher_retour(écran, :rafraîchir)
    {:noreply, écran}
  end

  defp afficher_écran(écran, message \\ "", seulement_prompt \\ false) do
    module = écran.__struct__
    titre = module.titre(écran)
    texte = module.texte(écran)
    prompt = module.prompt(écran)

    complet = """
    #{(seulement_prompt && "") || titre}

    #{(seulement_prompt && "") || String.trim(texte)}
    #{message}
    #{prompt}
    """

    IO.puts(String.trim(complet))
    écran
  end

  defp afficher_retour(écran, :silence), do: écran

  defp afficher_retour(écran, :prompt) do
    afficher_écran(écran, "", true)
  end

  defp afficher_retour(écran, :rafraîchir) do
    afficher_écran(écran)
  end

  defp afficher_retour(écran, {:silence, retour}) do
    IO.puts(retour)
    écran
  end

  defp afficher_retour(écran, {:prompt, retour}) do
    afficher_écran(écran, retour, true)
  end

  defp afficher_retour(écran, {:rafraîchir, retour}) do
    afficher_écran(écran, retour)
  end

  defp afficher_retour(_, écran) do
    afficher_retour(écran, :rafraîchir)
    écran
  end
end
